\ *****************************************************************************
\ hier kommt Quelltext fr Structs und Peripherals
\ *****************************************************************************
\
\ Structs gruppieren zusammengehrende Variablen und verbessern so Lesbarkeit
\ von Programmcode.
\ Prioritt des weiter unten stehenden Codes ist das Schreiben von lesbarem
\ Quelltext, nicht ein mglichst forthiges Herangehen ans Thema "Struct".
\
\ Peripherals sprechen Peripheriebausteine an, haben ansonsten aber das gleiche
\ Innenleben wie Structs.
\
\ Dieser Quelltext wurde auf SwiftForth 3.12 geschrieben und sollte fr
\ ein anderes 32-Bit Forth einfach anpassbar sein.
\ Im Prinzip muss IS-STRUCT? nur an die Stelle hinter die Abfrage, ob das neue
\ Wort vielleicht eine Zahl war, eingefgt werden.
\
\
\ Syntaxregeln
\ ------------
\ - die Felder werden durch einen Punkt getrennt: settings.win.position.top
\
\ - die Namen lassen sich Zweiteilen, das reduziert Schreibarbeit
\   Bsp: settings.win.position.   \ <- ab jetzt geht es um position, das ist ein _rect
\                          20 .left !
\                          50 .top !
\                         700 .right !
\
\ - beginnt der Structname mit einem bereits angelegten Struct (z.B. SETTINGS.win.),
\   ist die Adresse des Structs zur Compilezeit bekannt.
\   Achtung:
\   (   ) SETTINGS.win  ( addr )  hinterlsst die Adresse vom Feld "win"
\   (   ) SETTINGS.win. (   )     hat keinen Stackeffekt, weil es nur der erste Teil vom Namen ist
\
\ - beginnt der Structname mit einem Defining Struct (z.B. _infostruct.win.),
\   wird die Adresse des Structs zur Laufzeit auf dem Stack erwartet.
\   Achtung:
\   ( addr ) _infostruct.win  ( addr' )  hinterlsst die Adresse vom Feld "win"
\   ( addr ) _infostruct.win. (   )      gilt als der erste Teil vom Namen,
\                                        addr wird vom Stack genommen und gemerkt
\
\
\ Endet der Structname mit einem Punkt bedeutet das:
\   - es ist der erste Teil eines geteilten Structnamens
\   - die bis dahin berechnete Feldadresse wird gemerkt und steht dann
\     fr den hinteren Teil des Structnamens zur Verfgung
\ Beginnt der Structname mit einem Punkt bedeutet das:
\   - es ist der zweite Teil eines geteilten Structnamens
\   - die Berechnung der Feldadresse sttzt sich auf den ersten Teil des Structnamens
\
\ Die Namen der Structs drfen keinen Punkt enthalten.
\ Die Namen der Structmember drfen ebenfalls keinen Punkt enthalten.
\ Fr Structmember sind Forth-Wrter wie HOLD, LOAD, CR etc. zulssig, ebenso Ziffern.
\
\ Am Beispiel SETTINGS.win.position.top
\ -------------------------------------
\ Das Wort vor dem ersten Punkt steht im Dictionary, entweder als Defining Struct
\ oder als angelegtes Struct.
\ Wird das Wort gefunden, ist auch die zugehrige Structmemberlist bekannt.
\ Die Structmemberlist enthlt Namen und Offset.
\ Fr z.B. _rect ist die Structmemberlist:  left $0 top $4 right $8 bottom $C
\
\ Das Wort nach dem Punkt (bis zum nchsten Punkt) wird nicht im Dictionary gesucht,
\ sondern in der Structmemberlist des Worts von vor dem Punkt.
\ Auf diese Weise hangelt sich der Mechanismus weiter, bis der String aufgebraucht ist.
\ Das Ergebnis ist die Summe der einzelnen Offsets, anschlieend wird die
\ Structadresse addiert man erhlt die gesuchte Adresse vom Feld.
\
\
\ Definieren von Structs
\ ----------------------
\ STRUCT _rect
\ 	Int32  left
\ 	Int32  top
\ 	Int32  right
\ 	Int32  bottom
\ END-STRUCT
\
\ STRUCT _win
\ 	_windowplacement wplace
\ 	_rect   position
\ 	_rect   Clientrect
\ END-STRUCT
\
\ STRUCT _infostruct
\ 	Int32         version
\ 	_win          win
\ 	80 Bytes      dateiname
\   10 $VAR       $comport
\ END-STRUCT
\
\ Anlegen eines Structs:
\ ----------------------
\ _infostruct SETTINGS
\
\ Zugriff auf Structs:
\ --------------------
\ SETTINGS.version @
\ SETTINGS.$comport COUNT TYPE
\
\ Beispiel fr zweigeteilte Structnamen:
\ --------------------------------------
\ SETTINGS.win.position.            \ position ist ein _rect, ab jetzt
\ 20                   .left !      \ werden Structmember in _rect gesucht
\ 70                   .top !
\ 700                  .right !
\ 400                  .bottom !
\
\ Die Structadresse liegt auf dem Stack:
\ --------------------------------------
\ ... ( addr ) _infostruct. (   )  ... (   ) .version @ ( n )
\ ... ( addr ) _infostruct.win.position.left ( addr' ) @ ( n )
\ Vereinbarung:
\ Beginnt der Struct mit dem Namen eines Defining Struct,
\ dann wird die Structadresse auf dem Stack erwartet.
\
\ Kopieren von Structs:
\ ---------------------
\ from-struct. (   ) STRUCT> ( from count ) to-struct. ( from count ) >STRUCT (   )
\ Beide Structs wurden tunlichst vom gleichen Defining Struct erzeugt!
\ Punkt am Ende nicht vergessen!
\
\ Initialisieren von Structs:
\ ---------------------------
\ Structs fhren beim Anlegen das xt vom Datafield aus, so lassen sich
\ Felder wie z.B. Lngeninfos, die spter nicht mehr gendert werden,
\ automatisch initialisieren.
\ Siehe STRUCT-INIT mit Beispiel und ERASE-STRUCT
\
\
\
\ Die Structs hier liefern nur die Adresse der einzelnen Felder.
\ Was an dieser Adresse zu geschehen hat, ist Aufgabe des Programmierers.
\
\ Der "Trick" bei diesen Structs ist, dass die Structmember nicht im
\ Dictionary stehen. Fr den Interpreter sind die Structmember erstmal ein
\ unbekanntes Wort, das noch in IS-STRUCT? untersucht wird, bevor es
\ evtl. nach UNDEFINED weitergereicht wird.
\
\ Die Structs hier werden daran erkannt, dass (mindestens) ein Punkt im
\ String vorkommt (z.B. UART3.CTRL oder UART3. oder .CTRL oder QTMR1.2.HOLD).
\ STRUCT? drselt den String auf und berechnet die passende Adresse zum
\ gefundenen Ergebnis.
\
\ Das Berechnen der Adressen geschieht zur Compilezeit, zur Laufzeit wird
\ dann ein LITERAL ausgefhrt.
\ Ausnahme ist das bergeben der Structadresse auf dem Stack, dann wird zur
\ Laufzeit noch die Basisadresse geladen und eine Addition ausgefhrt.
\
\ Beim Definieren der Structs entsteht eine sog. Structmemberlist, die die
\ Namen der Structmember und deren Offset zur Startadresse enthlt.
\ Die Structmemberlist steht lesbar in ASCII im data-field des Defining Structs.
\ In Anlehnung an 'Defining Word' wird hier der Begriff 'Defining Struct' eingefhrt.
\ Gemeint ist damit die Definition eines Structs mit STRUCT Name .... END-STRUCT.
\ Erst der Aufruf von Name (z.B. _infostruct SETTINGS) legt dann ein Struct
\ im Arbeitsspeicher an.
\
\ Die Structmemberlist ist platzsparend und vergrert das Dictionary nicht unntig.
\ Structmember knnen auch den Namen von Forth-Wrtern haben (z.B. sind LOAD, HOLD, CR
\ beliebt bei Timern). Gleiche Namen in unterschiedlichen Structs an unterschiedlicher
\ Stelle (z.B. Reserved frs Alignment) sind mglich. Eine Zahl als Name fr ein
\ Structmember ist ebenfalls mglich.
\
\ Structs knnen Teil eines anderen Structs sein, das gilt auch fr Peripherals.
\ Ein Mischen von Structs und Peripherals ist nicht vorgesehen.
\ Structs werden an der nchsten freien Stelle im Arbeitsspeicher angelegt,
\ Peripherals zeigen auf eine feste Stelle im Adressraum und bekommen ihre
\ Adresse vom Programmierer zugewiesen.
\
\ *****************************************************************************

\ Gebrauch:
\
\ hier ist ein kurzes Beispiel zur Verwendung der Structs
\
\ STRUCT Header       \ <- definiere ein Struct namens Header
\ 	Int32 Command
\   Int32 ID
\   Int32 Len
\   Int32 Checksum
\ END-STRUCT
\
\ STRUCT Inputconfig
\   Header IHeader    \ <- baue ein Struct in ein anderes Struct ein
\   Int32 PreAmp
\   Int16 Filter
\   Int16 unused
\   Int32 Muxer
\ END-STRUCT
\
\ Inputconfig INPUTCONFIG_A       \ lege ein Struct an
\ Inputconfig INPUTCONFIG_B
\
\ 16 INPUTCONFIG_A.IHeader.ID !   \ schreibe ins Struct
\ INPUTCONFIG_B.Filter W@         \ lies aus dem Struct
\ Achtung, es werden nur die Adressen zurckgegeben,
\ der Datentyp liegt in der Verantwortung des Programmierers.
\

\ Datafield eines Defining Structs:
\ -----------------------------------------------------------------------------
\ |  Gre des Structs |     Adresse      |    xt fr die   | Structmemberlist
\ |      in Bytes      | Structmemberlist | Initialisierung |   ( in ASCII)
\ -----------------------------------------------------------------------------
\ ' INPUTCONFIG_A >BODY CELL+ @ $40 DUMP zeigt die Structmemberlist.
\ Die Lngeninfo fr das DUMP ist fr jede Structmemberlist natrlich unterschiedlich.
\ Die Structmemberlist endet mit einem 0x00.

\ Datafield eines durch ein Defining Struct angelegten Structs:
\ -----------------------------------------------------------------------------
\ |  Gre des Structs |     Adresse      |    xt fr die   |     Struct
\ |      in Bytes      | Structmemberlist | Initialisierung |
\ -----------------------------------------------------------------------------

\ Im Datafield steht zuerst die Gre des Structs in Bytes.
\ Dahinter kommt die Adresse der Structmemberlist.
\ Dahinter dann das xt zur Initialierung des Structs.
\ Dahinter dann die Daten bzw. die Structmemberlist bei einem Defining Struct.


VARIABLE SML        \ Structmemberlist
VARIABLE SXT        \ Struct-xt, zeigt auf die Adresse wo das xt steht
VARIABLE STRUCTBASE \ Startadresse des Structs
VARIABLE STRUCTFLAG
VARIABLE PREV.STATE
VARIABLE SPLITFLAG
VARIABLE DOT_FOUND
VARIABLE MEMBERSTR
VARIABLE MEMBERLEN
VARIABLE LISTSTR
VARIABLE LISTLEN

\ *****************************************************************************
\ hier kommen Hilfswrter

DECIMAL

\ Ersatz fr STRPCK, das einen String (c-addr u) in einen Counted String wandelt.
\ Soll einfach nur ein 'is redefined' oder 'undefined' vermeiden.
CREATE PACKBUFFER 256 ALLOT
: $PCK	( c-addr u -- ^addr )
255 MIN DUP PACKBUFFER C!
PACKBUFFER 1+ SWAP ( from to count ) CMOVE
PACKBUFFER ( ^addr )
;

\ *****************************************************************************
\ hier wird die Structmemberlist im data-field eines Defining-Structs aufgebaut

DECIMAL

: APPEND.NAME (   --   )
\ schreibe das nchste Wort als String in die Structmemberlist
BL WORD ( ^str ) COUNT 0 ?DO ( addr ) DUP C@ C, 1+ LOOP DROP (   )
;

\ der Offset wird als Radix-16 incl. Prfix $ geschrieben,
\ dann ist das Lesen unabhngig vom momentanen Inhalt von BASE
: APPEND.OFFS  ( addr -- addr )
\ schreibe den aktuellen Offset (Inhalt von addr) in die Structmemberlist
BASE @ >R HEX
DUP @ ( addr n )     \ lies aktuellen Offset
( addr n ) 0 <# BL HOLD #S [CHAR] $ HOLD BL HOLD #> ( addr c_addr c_len ) \ Offset als String
0 ?DO DUP C@ C, 1+ LOOP DROP   ( addr ) \ schreibe String in die Structmemberlist
R> BASE !
;

: APPEND.LINK	( addr pfa_struct -- addr )  \ Link ist Offset und Adresse der Structmemberlist
\ der Link wird als "Offset.Structmemberlist" abgelegt.
\ d.h. "Offset" plus "Punkt" plus "Structmemberlist"
\ und wird dann von STRUCTMEMBER? als Double bzw. Zeiger auf ein weiteres Struct erkannt
BASE @ >R HEX
DUP @ SWAP CELL+ @ ( addr size link ) THIRD @ SWAP ( addr size offs link )
0 <# BL HOLD #S 2DROP 0 [CHAR] . HOLD #S [CHAR] $ HOLD BL HOLD #> ( addr size c_addr c_len )
0 ?DO DUP C@ C, 1+ LOOP DROP ( addr size )  \ size ist die Gre des Structs
( addr size ) OVER +! ( addr )
R> BASE !
;

\ *****************************************************************************
\ hier kommen Wrter zum Definieren von Structs

VARIABLE STRUCTNEST

: CREATE-STRUCT   ( pfa -- addr )
STRUCTNEST @ 0=
IF ( pfa )       \ hier landet er, wenn ein Defining Struct ein neues Struct anlegt
  CREATE ( pfa ) \ im Datafield liegt zuerst die Lnge vom Struct, dann die Adresse der Memberlist
  DUP @ DUP , ( pfa len )                \ bertrage die Lnge
  OVER CELL+ @ ( pfa len memberlist ) ,  \ und die Adresse der Memberlist
  OVER 2 CELLS + @ ( pfa len xt ) ,      \ und das xt ins neue Struct
  HERE SWAP ( pfa addr len )             \ HERE ist die Stelle, wo das Struct beginnt
  ( pfa addr len ) 0 ?DO 0 C, LOOP       \ lege erstmal ein leeres Struct an
  ( pfa addr ) SWAP 2 CELLS + @ ( addr xt )  \ hole das xt fr die Initialisierung
  ( addr xt ) DUP 0<> IF ( addr xt ) EXECUTE (   ) ELSE 2DROP (   ) THEN \ das xt initialisiert dann das leere Struct
  DOES> 3 CELLS +                        \ gib zur Laufzeit die Startadresse des Structs
ELSE ( addr pfa_struct ) \ hier soll ein bereits definierter Struct Teil eines anderen Structs werden
  ( addr pfa_struct ) SWAP APPEND.NAME SWAP ( addr pfa_struct )
  ( addr pfa_struct ) APPEND.LINK ( addr )
THEN
;

: STRUCT       (   -- addr ) \ in addr steht die Lnge des Structs
1 STRUCTNEST !
CREATE HERE ( addr ) 0 , HERE 2 CELLS + , 0 , BL C,   \ hier landet er, wenn ein Struct angefangen wird: STRUCT Name ....
       \ pfa: +0:Structsize   +4:Memberlist   +8:xt   +12:Memberlist
       \ Structsize ist noch Null und die Memberlist beginnt an zweiter Stelle im Parameterfield von "Name"
       \ an dritter Stelle steht das xt, ist es ungleich Null, wird es beim Anlegen des Structs ausgefhrt.
DOES> CREATE-STRUCT                     \ hier landet er, wenn ein Struct Teil eines anderen Structs werden soll
;

: END-STRUCT   ( addr --   )
0 C, \ beende die Structmemberlist mit einem Nullbyte
0 STRUCTNEST !
DROP
;

\ Achtung, es gibt hier kein automatisches Alignment!
\ Beim Erstellen eines Structs selber drauf achten, dass Int32 u. Int16
\ an 4-Byte-Adressen bzw. 2-Byte-Adressen liegen.

: Int32        ( addr -- addr ) 		\ 32 Bit
DUP @ 1- 3 OR 1+ OVER ! \ Alignment
( addr ) APPEND.NAME APPEND.OFFS ( addr ) CELL OVER +! ( addr )
;

: Int16         ( addr -- addr ) 	\ 16 Bit
DUP @ 1- 1 OR 1+ OVER ! \ Alignment
( addr ) APPEND.NAME APPEND.OFFS ( addr ) 2 OVER +! ( addr )
;

: Byte					( addr -- addr ) 		\ 8 Bit
( addr ) APPEND.NAME APPEND.OFFS ( addr ) 1 OVER +! ( addr )
;

: Bytes         ( addr len -- addr ) \ len Bytes
0 MAX 10000 MIN
( addr len ) SWAP APPEND.NAME APPEND.OFFS SWAP ( addr len ) OVER +! ( addr )
;

: $VAR          ( addr len -- addr ) \ Stringvariable
\ legt eine Stringvariable als Counted String mit Platz fr len Bytes an
( addr len ) 1 MAX 255 MIN \ Lnge wg. Counted String begrenzen
SWAP DUP @ 1- 3 OR 1+ OVER ! SWAP ( addr len ) \ Inhalt von addr zeigt jetzt auf eine 4-Byte-Adresse
\ Achtung, an die Adresse 4 Bytes vor! dem Counted String muss beim Anlegen des Structs
\ die Lngeninfo len geschieben werden!
( addr len ) CELL THIRD +! \ schiebe Anfang des Counted String 4 Bytes weiter
( addr len ) SWAP APPEND.NAME APPEND.OFFS SWAP ( addr len )
( addr len' ) 1+ ( addr len'' ) OVER +!  \ plus 1 Byte fr das Countbyte
\ flle jetzt noch auf bis zur nchsten 4-Byte-Adresse
( addr ) DUP @ 1- 3 OR 1+ OVER ! ( addr )
;

: INIT.$VAR     ( n addr --   ) \ addr ist die Adresse der Stringvariablen
\ die Lngeninfo von Stringvariablen muss nach dem Anlegen vom Struct
\ 'von Hand' an die Stelle vor der Stringvariablen eingetragen werden, ansonsten
\ funktionieren $!, ERASE.$ u. BLANK.$ nicht, die verwenden nmlich den zur
\ Verfgung stehenden Platz
CELL- !
;

\ *****************************************************************************
\ Hier kommen Wrter fr den einfachen Zugriff auf Peripherie,
\ sinnvoll hauptschlich fr Mikrocontroller.
\ Ziel ist ein "lesbarer" Zugriff auf Peripherieregister durch weniger
\ Schreibarbeit.
\
\ Syntax am Beispiel UART: 0 UART3.CTRL !  oder  UART3.DATA @
\
\ Fr den Interpreter sind PERIPHs auch nur Structs, es gelten daher
\ die gleichen Syntaxregeln wie bei den Structs.
\ Ein Teilen der Registernamen wie bei den Structs (in UART3. und .CTRL)
\ ist ebenfalls mglich.

: CREATE-PERIPH		( addr_periph structinfo -- addr )
STRUCTNEST @ 0=
IF ( addr pfa )
	CREATE ( addr pfa )
	SWAP , ( pfa )
	CELL+ @ ( memberlist ) ,
	0 , \ an dieser Stelle ist bei den Structs das xt, bei PERIPH nicht verwendet
	DOES> @         \ gib zur Laufzeit die Startadresse der Peripherie
ELSE ( addr pfa ) \ hier soll ein Registerblock eingefgt werden,
	\ z.B. eine Registergruppe von einem Quad-Timer.
	\ Hole als Erstes mal die Registeradresse und schreibe sie nach addr
	SWAP ( pfa addr ) BL WORD ( pfa addr ^regaddr )
  COUNT NUMBER DPL @ 0< NOT IF DROP THEN ( pfa addr regaddr ) OVER !
  ( pfa addr ) APPEND.NAME SWAP ( addr pfa )
  ( addr pfa ) APPEND.LINK ( addr )
THEN
;

: PERIPH			(   -- addr ) \ addr dient als Zwischenspeicher fr die Registeradressen
1 STRUCTNEST !
CREATE HERE ( addr ) 0 , HERE 2 CELLS + , 0 , BL C,
DOES> CREATE-PERIPH    \ hier landet er, wenn ein PERIPH Teil eines anderen PERIPHs werden soll
;

: END-PERIPH   ( addr --   )
0 C, \ beende die Structmemberlist mit einem Nullbyte
0 STRUCTNEST !
DROP
;

: REG	( addr -- addr ) \ Verwendung: REG n NAME   n ist der Offset fr das Register NAME
BL WORD ( addr ^str ) COUNT NUMBER ( addr n | addr d ) DPL @ 0< NOT IF DROP THEN
( addr n ) OVER ! \ NUMBER fhrt auch ein ABORT aus, falls ^str keine Zahl ist
( addr ) APPEND.NAME APPEND.OFFS ( addr )
;

\ hier ein Beispiel fr einen Quad-Timer:
\ Group ist die Registergruppe fr einen Timer, der Quad-Timer hat 4 davon, der Controller hat 3 Quad-Timer
\ Zugriff auf Register HOLD fr Timer 2 von QuadTimer1 ist:  QTMR1.2.HOLD
\ Achtung, 16-Bit-Register!
\
\
\ PERIPH GROUP              \ definiere ein Peripheral
\ 	REG $00 COMP1           \ hinter REG steht der Offset zur Startadresse
\ 	REG $02 COMP2           \ hinter dem Offset steht dann der Registername
\ 	REG $04 CAPT
\ 	REG $06 LOAD            \ die Registernamen mit Offset landen
\ 	REG $08 HOLD            \ in der Memberlist von GROUP
\ 	REG $0A CNTR
\ 	REG $0C CTRL
\ 	REG $0E SCTRL
\ 	REG $10 CMPLD1
\ 	REG $12 CMPLD2
\ 	REG $14 CSCTRL
\ END-PERIPH
\
\ PERIPH TMR
\ 	GROUP $00 0             \ <- baue ein Peripheral in ein anderes ein
\ 	REG $1E ENBL
\ 	GROUP $20 1
\ 	GROUP $40 2
\ 	GROUP $60 3
\ END-PERIPH
\
\ 401DC000 TMR QTMR1        \ Quad-Timer 1 liegt an Adresse 0x401DC000
\ 401E0000 TMR QTMR2
\ 401E4000 TMR QTMR3
\
\ QTMR1.0.HOLD W@  liest im Quad-Timer1 das Hold-Register von Timer 0

\ hier ein paar mgliche Schreibweisen am Beispiel UART:
\
\ 4018C000 UART UART3
\
\ : INIT.UART3  (   --   )
\          0 UART3.CTRL !  \ disable Sender und Empfnger
\ 0x0402008B UART3.BAUD !  \ setze Oversampling und Baudrate
\ 0x000C0000 UART3.CTRL !  \ enable Sender und Empfnger
\ ;
\
\ : INIT.UART3  (   --   )
\       UART3.  (   )      \ ab hier wird in der Memberlist von UART3 gesucht
\ die Adresse von UART3 ist bekannt und wird fr die Adressberechnung der
\ Register verwendet
\          0 .CTRL !
\ 0x0402008B .BAUD !
\ 0x000C0000 .CTRL !
\ ;
\
\ : INIT.UART  ( addr --   )  \ addr ist die Basisadresse eines UARTs
\ ( addr ) UART.  (   )  \ ab hier wird in der Memberlist von UART gesucht
\ UART ist ein Defining Peripheral, dort ist nur die Struktur hinterlegt,
\ aber keine Adresse. Deshalb wird davon ausgegangen, dass die Adresse auf
\ dem Stack liegt und nur noch der Offset addiert werden muss.
\          0 .CTRL !
\ 0x0402008B .BAUD !
\ 0x000C0000 .CTRL !
\ ;

\ *****************************************************************************
\ hier wird untersucht, ob ein String evtl. ein Struct bzw. ein Teil von
\ einem Struct ist

HEX

\ in Anlehnung an 'Defining Word' wird hier der Begriff 'Defining Struct' eingefhrt.
\ Gemeint ist damit die Definition eines Structs mit STRUCT Name .... END-STRUCT.
\ Erst der Aufruf von Name erzeugt dann ein Struct im Arbeitsspeicher.
\ Defining Structs lassen sich dadurch erkennen, dass die Structmemberlist
\ an der Data-Field Address plus 12 beginnt.
\ Von den Defining Structs angelegte Structs zeigen lediglich auf diese Structmemberlist,
\ die Differenz von ihrer eigenen Data-Field Address zur Structmemberlist ist dann grer.
\ Diese Unterscheidung wird dann im weiteren Verlauf noch wichtig.

: DEFINING_STRUCT? ( xt -- flag )
>BODY ( dfa ) DUP CELL+ @ ( dfa list ) SWAP - 0C =
;

\ SPLIT_STRING ( addr len char ) sucht im String nach char und teilt den String
\ in zwei Teilstrings ( addr2 len2 addr1 len1 ).
\ Es gibt 3 Flle:
\ 1) char ist mitten im String -> addr1_len1 ist der String bis zu char, addr2_len2 ist der restliche String hinter char
\ 2) char ist am Anfang vom String -> addr1_len1 hat die Lnge Null, addr2_len2 ist der restliche String hinter char
\ 3) char ist am Ende vom String -> addr1_len1 ist der String bis zu char, addr2_len2 hat die Lnge Null
\ Fall 3) gilt auch, wenn gar kein char im String war
\ DOT_FOUND wird TRUE, wenn ein Punkt im String gefunden wurde
: SPLIT_STRING ( addr len char -- addr2 len2 addr1 len1 )
ROT DUP 2SWAP ( addr1 addr1 len1 char ) SCAN ( addr1 addr2 len2 ) DUP 0<> DOT_FOUND !
OVER 2SWAP ( len2 addr2 addr1 addr2 ) OVER - 2SWAP ( addr1 len1 len2 addr2 )
1+ SWAP 1- 0 MAX
2SWAP
;

\ die Structmemberlist liegt im Datafield des Defining Struct
: STRUCTMEMBER? ( addr len -- addr len false | offset true )
\ die Structmember sehen folgendermaen aus:
\ - ... Name $offset ... , der Offset ist explizit in Hex, dann ist die Base eindeutig
\ - ... Name $offset.address ... , ist die Zahl hinter Name ein Double, dann ist Name ein Struct
\   Achtung: vor dem Dezimalpunkt kommt der Offset, dahinter dann die Adresse der neuen Memberlist
\            weitere Structmember werden ab sofort in der neuen Memberlist gesucht
0 SPLITFLAG !
SML @ 0<> \ gibt es eine Structmemberlist?
IF
	( addr len ) 2DUP ( addr len addr len ) 0 -ROT ( addr len offs addr len )
	FALSE -ROT ( addr len offs flag addr len )
	BEGIN ( addr len offs flag addr len )
		DUP 0>
	WHILE \ lse den String (addr len) so lange auf, bis len auf Null runter ist
		( addr len offs flag addr len ) ROT DROP FALSE -ROT ( addr len offs flag' addr len )
		( addr len offs flag addr len ) [CHAR] . SPLIT_STRING ( addr len offs flag addr2 len2 addr1 len1 )
		DOT_FOUND @ SPLITFLAG ! \
		SML @ ZCOUNT ( addr len offs flag addr2 len2 addr1 len1 addr3 len3 ) \ addr:Memberlist
		( addr len offs flag addr2 len2 addr1 len1 addr3 len3 ) LISTLEN ! LISTSTR ! MEMBERLEN ! MEMBERSTR !
		BEGIN
			( addr len offs flag addr2 len2 ) LISTSTR @ LISTLEN @ MEMBERSTR @ MEMBERLEN @
			SEARCH       ( addr len offs flag addr2 len2 addr4 len4 flag )
			IF \ zweites Wort wurde in der Memberlist gefunden
				( addr len offs flag addr2 len2 addr4 len4 )
				OVER MEMBERLEN @ + C@ BL = \ ist hinter dem String kein Blank, hat man nur einen
				                           \ lngeren String gefunden, bei dem zufllig die ersten Zeichen passen
				( addr len offs flag addr2 len2 addr4 len4 flag )
				THIRD 1- C@ BL = AND \ ist vor dem String kein Blank, hat man den hinteren Teil eines lngeren Strings gefunden
				IF \ der String wurde in der Memberlist gefunden, hinter dem String steht der Offset
					( addr len offs flag addr2 len2 addr4 len4 ) \ suche das nchste Zeichen ungleich Blank
					SWAP MEMBERLEN @ + SWAP MEMBERLEN @ - 0 MAX ( addr len offs flag addr2 len2 addr4' len4' )
					BEGIN ( ... addr4' len4' )
						OVER C@ BL = OVER 0> AND
					WHILE
						SWAP 1+ SWAP 1-
					REPEAT ( addr len offs flag addr2 len2 addr4'' len4'' )
					\ addr4 zeigt jetzt auf die Stelle, wo der Offset beginnt
					BL ( ... addr4'' len4'' char ) SPLIT_STRING 2SWAP 2DROP ( addr5 len5 ) NUMBER
					( addr len offs flag addr2 len2 n|d )
					\ Single: der Offset im Struct  Double: Zeiger auf ein Struct im Struct
					DPL @ 0<
					IF \ NUMBER hat ein Single zurckgegeben, dann ist der Name dazu
						\ einfach ein Element im Struct und die Suche endet hier
						( addr len offs flag addr2 len2 n )
						NIP NIP NIP ( addr len offs n ) + ( addr len offs' ) TRUE 0.
						( addr len offs' flag dummyaddr dummylen )
					ELSE \ NUMBER hat ein Double zurckgegeben, dann ist der Name dazu ein Struct
						\ und es muss in der dazu passenden Structmemberlist weitergesucht werden
						( addr len offs flag addr2 len2 d )
						\ vor dem Dezimalpunkt: Offset, dahinter die Adresse der neuen Structmemberlist
						\ ein Shiften von Doubles fehlt in SwiftForth leider, deshalb hier der Umweg ber Division
						1 DPL @ CELLS LSHIFT UM/MOD ( addr len offs flag addr2 len2 list n )
						SWAP SML !	( addr len offs flag addr2 len2 n ) \ schreibe Adresse der Structmemberlist nach SML
						SML @ ZCOUNT ( ... addr3 len3 ) LISTLEN ! LISTSTR ! \ es gilt jetzt die Memberlist vom neuen Struct
						SWAP >R SWAP >R SWAP DROP TRUE >R ( addr len offs n ) + R> R> R> ( addr len offs' flag addr2 len2 )
						\ (... addr2 len2) ist der restliche String, ist len2 Null, ist der Suchstring abgearbeitet
						DUP 0= IF 2DROP DROP ( addr len offs' ) TRUE 0. THEN
						\ hat er beim letzten SPLIT_STRING noch einen Punkt gefunden, dann war der
						\ Punkt das letzte Zeichen
						( addr len offs' flag addr len )
						\ mit dem unterschobenen TRUE verlsst er den BEGIN...UNTIL-Teil und
						\ luft wieder in den WHILE...REPEAT-Teil
					THEN
				ELSE \ der String geht hinter dem Suchstring weiter -> das war noch nichts
					\ berspringe den gefundenen String und suche weiter
					( addr len offs flag addr2 len2 addr4 len4 )
					BEGIN ( ... addr4 len4 )   \ suche das nchste Zeichen ungleich Blank
						OVER C@ BL <> OVER 0> AND
					WHILE
						SWAP 1+ SWAP 1-
					REPEAT ( addr len offs flag addr2 len2 addr4' len4' ) LISTLEN ! LISTSTR !
					( addr len offs flag addr2 len2 )
				THEN
			ELSE \ der Suchstring wurde in der Memberlist nicht gefunden -> Abbruch
				( addr len offs flag addr2 len2 addr4 len4 )
				2DROP 2DROP DROP FALSE ( addr len offs flag )
				0. ( addr len offs flag 0 0 )
			THEN
			( addr len offs' flag addr len )
			\ springe zurck zu Search bis entweder TRUE oder addr u. len beide Null sind
			THIRD TRUE = THIRD THIRD OR 0= OR
		UNTIL
	REPEAT
	( addr len offs flag addr 0 )
	2DROP
	IF ( addr len offs )
		NIP NIP TRUE ( offs true )
	ELSE ( addr len offs )
		DROP FALSE
	THEN
ELSE \ Structmemberlist ist leer, dann ist es auch kein Structmember
	( addr len ) FALSE
THEN
;

\ Idealerweise ist (c-addr u) ein String wie z.B INPUTCONFIG_A.IHeader.ID
\ Das Wort vor dem ersten Punkt wird im Dictionary gefunden und gibt
\ die Basisadresse des Structs bzw. Peripherals zurck.
\ Dabei wird auch gleich die Adresse der Structmemberlist gefunden.
\ Der Rest rechts vom ersten Punkt wird von STRUCTMEMBER? untersucht.
\ Der nchste Teilstring (bis zum nchsten Punkt) wird in der
\ Structmemberlist gesucht. Trifft er dabei auf einen eingebetteten Struct,
\ wird dann in dessen Structmemberlist weitergesucht.
\ Das wird so lange wiederholt, bis der String aufgebraucht ist.

: STRUCT? ( c-addr u -- lit flag ) \ es gibt 7 Flags: FALSE, 1...6
STATE @ PREV.STATE !
( addr len ) [CHAR] . SPLIT_STRING ( addr2 len2 addr1 len1 )
THIRD 0<>
IF
	DUP 0<>
	IF  \ "Normalfall", vor und hinter dem ersten Punkt gibt es einen String
   	( addr2 len2 addr1 len1 ) $PCK ( addr2 len2 ^str1 )
		FIND  ( addr2 len2 ^str1 false | addr2 len2 xt1 1 | addr2 len2 xt1 -1 )
		0<>
		IF \ ^str1 wurde im Vocabulary gefunden
			DUP >BODY ( addr2 len2 xt1 dfa )
			CELL+ @ ( addr2 len2 xt1 sml ) SML ! \ sml: Structmemberlist
			( addr2 len2 xt1 ) ROT ROT ( xt1 addr2 len2 )
			STRUCTMEMBER?    ( xt1 addr2 len2 false | xt1 offset true )
			IF \ Structmember addr2_len2 wurde gefunden
				( xt1 offset ) OVER DEFINING_STRUCT?
				IF \ xt1 gehrt zu einem DEFINING_STRUCT und wrde ein neues Struct beginnen.
					\ xt1 darf hier also nicht ausgefhrt werden, hier ist nur die
					\ Structmemberlist wichtig
					( xt1 offset ) NIP
					0 STRUCTBASE !
					SPLITFLAG @ \ war ein Punkt am Ende?
					IF \ Structbase liegt erst zur Laufzeit auf dem Stack
						\ und der offset plus die Structbase vom Stack ergeben eine neue Structbase
						( offset ) 6 ( offset flag ) \ flag=6 -> Structbase vom Stack plus offset ergibt neue Structbase
					ELSE \ Structbase liegt erst zur Laufzeit auf dem Stack
						\ der offset muss dann noch addiert werden
						( offset ) 5 ( offset flag ) \ flag=5 -> Structbase vom Stack plus offset ergibt Adresse
					THEN
				ELSE
					( xt1 offset )
					0 STATE ! \ stelle sicher, dass xt1 gleich interpretiert wird
					\ Achtung, xt1 muss!! eine Adresse auf dem Stack hinterlassen
					SWAP ( offset xt1 ) EXECUTE ( offset addr ) + ( lit )
					SPLITFLAG @ \ war ein Punkt am Ende?
					IF \ dann kommt lit nicht auf den Stack, sondern in die Structbase
						( lit ) DUP STRUCTBASE !
						2 ( lit flag ) \ flag=2 -> keine Aktion erforderlich
					ELSE
						( lit ) DUP STRUCTBASE !
						( lit ) 3  ( lit flag )  \ flag=3 -> lit soll auf den Stack
					THEN
					( lit flag )
				THEN
			ELSE \ addr2 len2 wurde nicht in der Structmemberlist gefunden
				( xt1 addr2 len2 )
				\ hier kann man jetzt evtl. noch im Dictionary suchen
				\ oder versuchen, ob es eine Zahl ist.
				\ der Einfachheit halber wird hier aber abgebrochen
				2DROP DROP 0 FALSE ( 0 false )
			THEN
		ELSE \ ^str1 wurde nicht gefunden
			( addr2 len2 ^str1 ) 2DROP DROP 0 FALSE ( 0 false )
		THEN
	ELSE \ String beginnt mit einem Punkt, string1 ist leer
		( addr2 len2 addr1 len1=0 )
		SML @ 0<>
		IF \ es wurde zuvor eine Structmemberlist gefunden
			SML @ >R
			2DROP ( addr2 len2 )
			STRUCTMEMBER? ( addr2 len2 -- addr2 len2 false | offset true )
			IF \ der String wurde in der Structmemberlist gefunden
				( offset )
				STRUCTBASE @ ( offset sb )
				DUP 0<>
				IF   \ zuvor wurde ein Struct im Arbeitsspeicher gefunden
					( offset sb ) + ( lit ) 3 ( lit flag )
				ELSE \ es wurde ein Defining Struct gefunden,
					\ die Adresse des Structs hat auf dem Stack zu liegen!
					( offset sb ) DROP  ( lit )
					1 ( lit flag )  \ flag=1 ->	lit muss auf eine Adresse addiert werden, die bereits auf dem Stack liegt
				THEN
			ELSE \ der String wurde nicht gefunden
		 		( addr len )
		 		2DROP 0 FALSE
			THEN
			R> SML !
		ELSE \ es gibt keine Structmemberlist, in der gesucht werden kann
			( addr2 len2 addr1 len1 )
			2DROP 2DROP
			0 FALSE
		THEN
	THEN
ELSE \ string1 endet vermutlich mit einem Punkt, string2 ist leer
	( addr2 len2=0 addr1 len1 )
	DOT_FOUND @
	0<>
	IF \ ja, es gibt einen Punkt am Stringende
		( addr2 len2 addr1 len1 ) 2SWAP 2DROP $PCK ( ^str1 )
		FIND ( ^str1 false | xt1 1 | xt1 -1 )
		0<>
		IF \ der Name wurde gefunden, das hat jetzt ein Struct zu sein!!!
			( xt1 ) DUP >BODY CELL+ DUP @ SML ! CELL+ SXT ! \ sml: Structmemberlist  sxt: Struct-xt
		  ( xt1 ) DUP DEFINING_STRUCT?
		  IF \ ein Defining-Struct darf nicht ausgefhrt werden, das will einen Dictionary-Eintrag erzeugen
		  	\ findet er ein Defining Struct, lautet die Vereinbarung: die Structadresse liegt auf Stack
		  	( xt1 ) DROP
		  	(   ) 0 STRUCTBASE !
		  	0 ( lit ) 4 ( lit flag ) \ flag=4 -> Structbase liegt erst zur Laufzeit auf dem Stack
		  ELSE \ es wird die Basisadresse des Structs als lit zurckgegeben
		  	0 STATE ! \ sicherheitshalber
		  	( xt1 ) EXECUTE ( lit ) \ ein Struct hinterlsst seine Basisadresse auf dem Stack
		  	( lit ) DUP STRUCTBASE !
		  	( lit ) 2 ( lit flag ) \ flag=2 -> keine Aktion erforderlich
		  THEN
		ELSE \ der Name war nicht zu finden, das ist natrlich auch ein Fehler
			( ^str ) DROP
			0 SML ! \ setze Structmemberlist auf Null
			0 SXT ! \ setze Struct-xt auf Null
			0 STRUCTBASE !
			(   ) 0 FALSE ( lit flag )
		THEN
	ELSE \ es gab gar keinen Punkt am Stringende, dann wars auch kein Struct
		( addr2 len2 addr1 len1 ) 2DROP 2DROP
		(   ) 0 FALSE ( lit flag )
	THEN
THEN
PREV.STATE @ STATE !
;

\ IS-STRUCT? wird aufgerufen, wenn das Wort nicht im Dictionary gefunden wurde.
\ STRUCT? prft, ob das unbekannte Wort vielleicht zu einem Struct gehrt.
: IS-STRUCT? ( c-addr len false -- c-addr len false | true )
DROP 2DUP ( c-addr len c-addr len )
STRUCT? ( c-addr len lit flag )   \ hier passiert die Magie, alles danach wertet nur flag aus
DUP 0>  \ flag von STRUCT? ist 0, 1, 2 oder 3
IF \ STRUCT? hat ein Struct gefunden
	( c-addr len lit flag ) 2SWAP 2DROP
	( lit flag )
	STATE @ 0=
	IF \ es wird Interpretiert
		( lit flag )
		CASE ( lit )
			1 OF \ die Basisadresse liegt in STRUCTBASE, addiere lit
					( lit ) STRUCTBASE @	+ ( addr )
				ENDOF
			2 OF \ kein Stackeffekt, es ging nur um die Structmemberlist
					( lit ) DROP
				ENDOF
			3 OF \ hinterlasse lit auf dem Stack
					( lit )
				ENDOF
			4 OF \ Structbase liegt auf dem Stack,
					 \ Anwendung: (addr) DefiningStruct.
					 \ Achtung: den Punkt am Ende nicht vergessen, sonst wird DefiningStruct ausgefhrt!
					 ( addr lit ) \ addr lag schon vor Aufruf von IS-STRUCT? auf dem Stack
					 DROP \ weg mit dem Dummy-lit von STRUCT?
					( baseaddr ) STRUCTBASE !
				ENDOF
			5 OF \ Structbase liegt schon auf dem Stack, lit ist der Offset dazu
					( baseaddr offset ) OVER STRUCTBASE !
					+ ( addr ) \ hinterlasse die Adresse des Structmembers auf dem Stack
				ENDOF
			6 OF \ Punkt am Ende, Structbase vom Stack plus Offset ergibt neue Structbase
					( baseaddr offset ) + STRUCTBASE !
					(   )
				ENDOF
			\ dieser Fall sollte eigentlich nicht vorkommen
			( lit flag ) 2DROP
		ENDCASE
		(   )
	ELSE \ es wird compiliert
		( lit flag )
		CASE ( lit )
			1 OF \ Zweiter Teil eines zweiteiligen Structnamens (beginnt mit einem Punkt).
					\ Die Basisadresse liegt in STRUCTBASE (auch zur Laufzeit)
					\ Frs Compilieren heisst das: Compiliere das lit und ein STRUCTBASE @ +
					( lit ) POSTPONE LITERAL POSTPONE STRUCTBASE POSTPONE @ POSTPONE +
				ENDOF
			2 OF \ Erster Teil eines zweiteiligen Structnamens (endet mit einem Punkt).
					\ Ist der erste Teil des Namens kein! Defining-Struct, ist die Basisadresse
					\ des Structs ist ab jetzt bekannt und wird auch zum Berechnen der Adressen
					\ der Structitems verwendet. Kein Stackeffekt!
					\ Rette Variableninhalte von STRUCTBASE u. SML von der Compilezeit in die Laufzeit
					( lit ) POSTPONE LITERAL POSTPONE STRUCTBASE POSTPONE !
					SML @ ( lit )  POSTPONE LITERAL POSTPONE SML POSTPONE !
				ENDOF
			3 OF \ compiliere die Adresse eines Structitems als lit
					( lit ) POSTPONE LITERAL
				ENDOF
			4 OF \ Erster Teil eines zweiteiligen Structnamens, es war allerdings ein Defining Struct.
					\ Die Basisadresse des Structs liegt zur Laufzeit auf dem Stack.
					\ Schreibe die Basisadresse nach STRUCTBASE, die wird dann fr Case 1 bentigt
					( addr lit ) DROP \ weg mit dem lit zur Compilezeit
					\ Schreibe die Basisadresse zur Laufzeit nach STRUCTBASE
					( addr ) POSTPONE STRUCTBASE POSTPONE !
					\ rette die Structmemberlist von der Compilezeit in die Laufzeit
					SML @ ( lit )  POSTPONE LITERAL POSTPONE SML POSTPONE !
				ENDOF
			5 OF \ der Name beginnt als Defining Struct, geht dann weiter und endet ohne Punkt
					\ Die Basisadresse liegt zur Laufzeit auf dem Stack, da muss noch der Offset addiert werden
					( offset ) POSTPONE LITERAL
					\ zur Laufzeit liegen dann ( addr offset ) auf dem Stack
					POSTPONE OVER POSTPONE STRUCTBASE POSTPONE ! POSTPONE +
					\ rette die Structmemberlist von der Compilezeit in die Laufzeit
					SML @ ( lit )  POSTPONE LITERAL POSTPONE SML POSTPONE !
				ENDOF
			6 OF \ der Name beginnt als Defining Struct, geht dann weiter und endet mit einem Punkt.
					\ Die Basisadresse liegt zur Laufzeit auf dem Stack, der Offset wird addiert und
					\ gibt dann die neue Basisadresse
					( offset ) POSTPONE LITERAL
					\ zur Laufzeit liegen dann ( addr offset ) auf dem Stack
					POSTPONE + POSTPONE STRUCTBASE POSTPONE !
					\ rette die Structmemberlist von der Compilezeit in die Laufzeit
					SML @ ( lit )  POSTPONE LITERAL POSTPONE SML POSTPONE !
				ENDOF
			\ an diese Stelle sollte er eigentlich nicht kommen
			( lit flag ) 2DROP
		ENDCASE
		(   )
	THEN
	TRUE	( true )
ELSE \ c-addr war kein Struct
	( c-addr len lit flag )
	2DROP FALSE ( c-addr len false )
THEN
;

\ Hier wird IS-STRUCT? ins INTERPRET, genauer gesagt ins NOT-DEFINED von SwiftForth eingehngt
\ Achtung, SwiftForth prft erst ob der String eine Zahl ist, bevor es die
\ UNDEFINED-Chain abarbeitet.
 ' IS-STRUCT? UNDEFINED >CHAIN

\ *****************************************************************************
\ Hier kommen einige Hilfswrter, die Schreibarbeit im Quelltext reduzieren
\ *****************************************************************************
HEX
\ STRUCT> und >STRUCT werden zum Kopieren des Inhalts von einem Struct
\ zu einem anderen Struct verwendet.
\ Beide Structs mssen vom gleichen Defining Struct angelegt sein!!!
\ Gebrauch von STRUCT> und >STRUCT:
\ from_Structname. STRUCT> ( from count ) to_Structname. >STRUCT
\ Achtung, Punkt am Ende nicht vergessen!
\ Funktioniert nur fr Structs, nicht fr einzelne darin enthaltenen Daten.
\ Kann aber sehr wohl einen Teil-Struct in einen anderen Struct kopieren.
: STRUCT>		(   -- from count ) \ Gebrauch: from_Structname. STRUCT>
STRUCTBASE @ SML @ 0C - @
;

: >STRUCT   ( from count --   ) \ Gebrauch: to_Structname. >STRUCT
\ Achtung, Quelle und Lnge mssen schon auf dem Stack liegen
STRUCTBASE @ ( from count to ) SWAP CMOVE
;

: ERASE-STRUCT (   --   ) \ Anwendung: Structname. ERASE-STRUCT  <- Punkt nicht vergessen!
\ lscht und initialisiert das Struct
\ structbase minus 0x0C: Structlnge
\ structbase minus   08: Adresse der Structmemberlist
\ structbase minus   04: falls ungleich Null: xt der Initialisierung
\ DUP DUP $0C - @ 0 MAX ( addr addr len ) ERASE ( addr )
\ DUP CELL- @ ( addr xt ) DUP 0<> IF ( addr xt ) EXECUTE (   ) ELSE 2DROP (   ) THEN
STRUCTBASE @ SML @ 0C - @ 0 MAX ERASE
STRUCTBASE @ SML @ CELL- @ ( addr xt ) DUP 0<> IF ( addr xt ) EXECUTE (   ) ELSE 2DROP (   ) THEN
;

\ Frisch angelegte Structs enthalten erstmal nur Nullen.
\ Ermglicht das automatische Initialisieren frisch angelegter Structs.
\ Siehe auch CREATE-STRUCT, die Zeile vor DOES>
: STRUCT-INIT (   --   )
\ schreibt das zuletzt definierte Wort in das xt-Feld vom im Wort
\ verwendeten Defining Struct.
\ Damit werden alle! mit diesem Defining Struct angelegten Structs gleich initialisiert.
\ Steht (wie z.B. IMMEDIATE) gleich hinter dem Semicolon.
LAST @ ( ^str ) NAME> ( xt ) SXT @
( xt addr )
DUP 0<>
IF
	( xt addr ) !  \ in SXT liegt die Adresse vom xt-Feld vom letzten Struct
ELSE  \ Inhalt von SXT ist Null und damit kein xt
	( xt addr ) 2DROP
THEN
;
\ Beispiel:
\ STRUCT _rect
\ 	Int32  left
\ 	Int32  top
\ 	Int32  right
\ 	Int32  bottom
\ END-STRUCT
\
\ : init-rect ( addr --   ) \ zur Laufzeit liegt die Adresse vom Struct auf dem Stack!
\ ( addr ) _rect. (   )
\ 111 .left !
\ 222 .top !
\ 333 .right !
\ 444 .bottom !
\ ; STRUCT-INIT

\ sizeof geht davon aus, dass die Lngeninfo im Datafield an erster Stelle steht
: sizeof (   -- n ) \ Anwendung: sizeof NAME    Achtung, Struct muss im Dictionary stehen
STATE @ 0=
IF \ es wird interpretiert
 	BL WORD  ( ^addr ) \ hole den Namen zu sizeof
	FIND ( c-addr 0 | xt 1 | xt -1 )
	0<>
	IF ( xt )
		>BODY @
	ELSE ( c-addr ) \ Name wurde nicht gefunden
		DROP          \ dann gib
		-1            \ ein -1 (kleiner Null) zurck
	THEN
ELSE \ es wird compiliert
	BL WORD ( ^addr ) \ hole den Namen zu sizeof
	FIND ( c-addr 0 | xt 1 | xt -1 )
	0<>
	IF ( xt )
		>BODY @ POSTPONE LITERAL
	ELSE ( c-addr ) \ Name wurde nicht gefunden
		DROP
		ABORT" unknown name after SIZEOF "
	THEN
THEN
; IMMEDIATE

\ zum schnellen Rumspielen bei nderungen im Quelltext

\ STRUCT _point
\ 	Int32  x
\ 	Int32  y
\ END-STRUCT
\
\ STRUCT _rect
\ 	Int32  left
\ 	Int32  top
\ 	Int32  right
\ 	Int32  bottom
\ END-STRUCT
\
\ : init-rect ( addr --   )
\ ( addr ) _rect. (   )
\ 111 .left !
\ 222 .top !
\ 333 .right !
\ 444 .bottom !
\ ; STRUCT-INIT
\
\ STRUCT _wplacement
\ 	Int32  length
\ 	Int32  flags
\ 	Int32  showCmd
\ 	_point ptMinPosition
\ 	_point ptMaxPosition
\ 	_rect  position
\ 	_rect  pos2
\ END-STRUCT
\
\ : INIT-WP ( addr --   )
\ ( addr ) _wplacement.  (   )
\ 1 .length !
\ 2 .flags !
\ 3 .ptMinPosition.x !
\ 4 .position.left !
\ 5 .position.top !
\ 6 .pos2.left !
\ 7 .pos2.top !
\ ;  STRUCT-INIT
\
\ _wplacement xx



















